import * as assert from 'assert';
import test, { describe } from 'node:test';
import parse from '../parse.js';
import v1, { updateV1State } from '../v1.js';
const TIME = 1321644961388;
const RFC_V1 = 'c232ab00-9414-11ec-b3c8-9f68deced846';
const RFC_V1_BYTES = parse(RFC_V1);
const RFC_OPTIONS = {
    msecs: 0x17f22e279b0,
    nsecs: 0,
    clockseq: 0x33c8,
    node: Uint8Array.of(0x9f, 0x68, 0xde, 0xce, 0xd8, 0x46),
};
const RFC_RANDOM = Uint8Array.of(0, 0, 0, 0, 0, 0, 0, 0, RFC_OPTIONS.clockseq >> 8, RFC_OPTIONS.clockseq & 0xff, ...RFC_OPTIONS.node);
function compareV1TimeField(a, b) {
    a = a.split('-').slice(0, 3).reverse().join('');
    b = b.split('-').slice(0, 3).reverse().join('');
    return a < b ? -1 : a > b ? 1 : 0;
}
describe('v1', () => {
    test('v1 sort order (default)', () => {
        const ids = [v1(), v1(), v1(), v1(), v1()];
        const sorted = [...ids].sort(compareV1TimeField);
        assert.deepEqual(ids, sorted);
    });
    test('v1 sort order (time option)', () => {
        const ids = [
            v1({ msecs: TIME - 10 * 3600 * 1000 }),
            v1({ msecs: TIME - 1 }),
            v1({ msecs: TIME }),
            v1({ msecs: TIME + 1 }),
            v1({ msecs: TIME + 28 * 24 * 3600 * 1000 }),
        ];
        const sorted = [...ids].sort(compareV1TimeField);
        assert.deepEqual(ids, sorted);
    });
    test('v1(options)', () => {
        assert.equal(v1({ msecs: RFC_OPTIONS.msecs, random: RFC_RANDOM }), RFC_V1, 'minimal options');
        assert.equal(v1(RFC_OPTIONS), RFC_V1, 'full options');
    });
    test('v1(options) equality', () => {
        assert.notEqual(v1({ msecs: TIME }), v1({ msecs: TIME }), 'UUIDs with minimal options differ');
        assert.equal(v1(RFC_OPTIONS), v1(RFC_OPTIONS), 'UUIDs with full options are identical');
    });
    test('fills one UUID into a buffer as expected', () => {
        const buffer = new Uint8Array(16);
        const result = v1(RFC_OPTIONS, buffer);
        assert.deepEqual(buffer, RFC_V1_BYTES);
        assert.strictEqual(buffer, result);
    });
    test('fills two UUIDs into a buffer as expected', () => {
        const buffer = new Uint8Array(32);
        v1(RFC_OPTIONS, buffer, 0);
        v1(RFC_OPTIONS, buffer, 16);
        const expectedBuf = new Uint8Array(32);
        expectedBuf.set(RFC_V1_BYTES);
        expectedBuf.set(RFC_V1_BYTES, 16);
        assert.deepEqual(buffer, expectedBuf);
    });
    test('v1() state transitions', () => {
        const PRE_STATE = {
            msecs: 10,
            nsecs: 20,
            clockseq: 0x1234,
            node: Uint8Array.of(0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc),
        };
        const tests = [
            {
                title: 'initial state',
                state: {},
                now: 10,
                expected: {
                    msecs: 10,
                    nsecs: 0,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
            {
                title: 'same time interval',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs,
                expected: {
                    ...PRE_STATE,
                    nsecs: 21,
                },
            },
            {
                title: 'new time interval',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs + 1,
                expected: {
                    ...PRE_STATE,
                    msecs: PRE_STATE.msecs + 1,
                    nsecs: 0,
                },
            },
            {
                title: 'same time interval (nsecs overflow)',
                state: { ...PRE_STATE, nsecs: 9999 },
                now: PRE_STATE.msecs,
                expected: {
                    ...PRE_STATE,
                    nsecs: 0,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
            {
                title: 'time regression',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs - 1,
                expected: {
                    ...PRE_STATE,
                    msecs: PRE_STATE.msecs - 1,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
        ];
        for (const { title, state, now, expected } of tests) {
            assert.deepStrictEqual(updateV1State(state, now, RFC_RANDOM), expected, `Failed: ${title}`);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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