"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert = require("assert");
const node_test_1 = require("node:test");
const parse_js_1 = require("../parse.js");
const v1_js_1 = require("../v1.js");
const TIME = 1321644961388;
const RFC_V1 = 'c232ab00-9414-11ec-b3c8-9f68deced846';
const RFC_V1_BYTES = (0, parse_js_1.default)(RFC_V1);
const RFC_OPTIONS = {
    msecs: 0x17f22e279b0,
    nsecs: 0,
    clockseq: 0x33c8,
    node: Uint8Array.of(0x9f, 0x68, 0xde, 0xce, 0xd8, 0x46),
};
const RFC_RANDOM = Uint8Array.of(0, 0, 0, 0, 0, 0, 0, 0, RFC_OPTIONS.clockseq >> 8, RFC_OPTIONS.clockseq & 0xff, ...RFC_OPTIONS.node);
function compareV1TimeField(a, b) {
    a = a.split('-').slice(0, 3).reverse().join('');
    b = b.split('-').slice(0, 3).reverse().join('');
    return a < b ? -1 : a > b ? 1 : 0;
}
(0, node_test_1.describe)('v1', () => {
    (0, node_test_1.default)('v1 sort order (default)', () => {
        const ids = [(0, v1_js_1.default)(), (0, v1_js_1.default)(), (0, v1_js_1.default)(), (0, v1_js_1.default)(), (0, v1_js_1.default)()];
        const sorted = [...ids].sort(compareV1TimeField);
        assert.deepEqual(ids, sorted);
    });
    (0, node_test_1.default)('v1 sort order (time option)', () => {
        const ids = [
            (0, v1_js_1.default)({ msecs: TIME - 10 * 3600 * 1000 }),
            (0, v1_js_1.default)({ msecs: TIME - 1 }),
            (0, v1_js_1.default)({ msecs: TIME }),
            (0, v1_js_1.default)({ msecs: TIME + 1 }),
            (0, v1_js_1.default)({ msecs: TIME + 28 * 24 * 3600 * 1000 }),
        ];
        const sorted = [...ids].sort(compareV1TimeField);
        assert.deepEqual(ids, sorted);
    });
    (0, node_test_1.default)('v1(options)', () => {
        assert.equal((0, v1_js_1.default)({ msecs: RFC_OPTIONS.msecs, random: RFC_RANDOM }), RFC_V1, 'minimal options');
        assert.equal((0, v1_js_1.default)(RFC_OPTIONS), RFC_V1, 'full options');
    });
    (0, node_test_1.default)('v1(options) equality', () => {
        assert.notEqual((0, v1_js_1.default)({ msecs: TIME }), (0, v1_js_1.default)({ msecs: TIME }), 'UUIDs with minimal options differ');
        assert.equal((0, v1_js_1.default)(RFC_OPTIONS), (0, v1_js_1.default)(RFC_OPTIONS), 'UUIDs with full options are identical');
    });
    (0, node_test_1.default)('fills one UUID into a buffer as expected', () => {
        const buffer = new Uint8Array(16);
        const result = (0, v1_js_1.default)(RFC_OPTIONS, buffer);
        assert.deepEqual(buffer, RFC_V1_BYTES);
        assert.strictEqual(buffer, result);
    });
    (0, node_test_1.default)('fills two UUIDs into a buffer as expected', () => {
        const buffer = new Uint8Array(32);
        (0, v1_js_1.default)(RFC_OPTIONS, buffer, 0);
        (0, v1_js_1.default)(RFC_OPTIONS, buffer, 16);
        const expectedBuf = new Uint8Array(32);
        expectedBuf.set(RFC_V1_BYTES);
        expectedBuf.set(RFC_V1_BYTES, 16);
        assert.deepEqual(buffer, expectedBuf);
    });
    (0, node_test_1.default)('v1() state transitions', () => {
        const PRE_STATE = {
            msecs: 10,
            nsecs: 20,
            clockseq: 0x1234,
            node: Uint8Array.of(0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc),
        };
        const tests = [
            {
                title: 'initial state',
                state: {},
                now: 10,
                expected: {
                    msecs: 10,
                    nsecs: 0,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
            {
                title: 'same time interval',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs,
                expected: {
                    ...PRE_STATE,
                    nsecs: 21,
                },
            },
            {
                title: 'new time interval',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs + 1,
                expected: {
                    ...PRE_STATE,
                    msecs: PRE_STATE.msecs + 1,
                    nsecs: 0,
                },
            },
            {
                title: 'same time interval (nsecs overflow)',
                state: { ...PRE_STATE, nsecs: 9999 },
                now: PRE_STATE.msecs,
                expected: {
                    ...PRE_STATE,
                    nsecs: 0,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
            {
                title: 'time regression',
                state: { ...PRE_STATE },
                now: PRE_STATE.msecs - 1,
                expected: {
                    ...PRE_STATE,
                    msecs: PRE_STATE.msecs - 1,
                    clockseq: RFC_OPTIONS.clockseq,
                    node: RFC_OPTIONS.node,
                },
            },
        ];
        for (const { title, state, now, expected } of tests) {
            assert.deepStrictEqual((0, v1_js_1.updateV1State)(state, now, RFC_RANDOM), expected, `Failed: ${title}`);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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