"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert = require("assert");
const node_test_1 = require("node:test");
const parse_js_1 = require("../parse.js");
const stringify_js_1 = require("../stringify.js");
const v7_js_1 = require("../v7.js");
const RFC_V7 = '017f22e2-79b0-7cc3-98c4-dc0c0c07398f';
const RFC_V7_BYTES = (0, parse_js_1.default)('017f22e2-79b0-7cc3-98c4-dc0c0c07398f');
const RFC_MSECS = 0x17f22e279b0;
const RFC_SEQ = (0x0cc3 << 20) | (0x98c4dc >> 2);
const RFC_RANDOM = Uint8Array.of(0x10, 0x91, 0x56, 0xbe, 0xc4, 0xfb, 0x0c, 0xc3, 0x18, 0xc4, 0x6c, 0x0c, 0x0c, 0x07, 0x39, 0x8f);
(0, node_test_1.describe)('v7', () => {
    (0, node_test_1.default)('subsequent UUIDs are different', () => {
        const id1 = (0, v7_js_1.default)();
        const id2 = (0, v7_js_1.default)();
        assert.ok(id1 !== id2);
    });
    (0, node_test_1.default)('explicit options.random and options.msecs produces expected result', () => {
        const id = (0, v7_js_1.default)({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        });
        assert.strictEqual(id, RFC_V7);
    });
    (0, node_test_1.default)('explicit options.rng produces expected result', () => {
        const id = (0, v7_js_1.default)({
            rng: () => RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        });
        assert.strictEqual(id, RFC_V7);
    });
    (0, node_test_1.default)('explicit options.msecs produces expected result', () => {
        const id = (0, v7_js_1.default)({
            msecs: RFC_MSECS,
        });
        assert.strictEqual(id.indexOf('017f22e2'), 0);
    });
    (0, node_test_1.default)('fills one UUID into a buffer as expected', () => {
        const buffer = new Uint8Array(16);
        const result = (0, v7_js_1.default)({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer);
        (0, stringify_js_1.default)(buffer);
        assert.deepEqual(buffer, RFC_V7_BYTES);
        assert.strictEqual(buffer, result);
    });
    (0, node_test_1.default)('fills two UUIDs into a buffer as expected', () => {
        const buffer = new Uint8Array(32);
        (0, v7_js_1.default)({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer, 0);
        (0, v7_js_1.default)({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer, 16);
        const expected = new Uint8Array(32);
        expected.set(RFC_V7_BYTES);
        expected.set(RFC_V7_BYTES, 16);
        assert.deepEqual(buffer, expected);
    });
    (0, node_test_1.default)('lexicographical sorting is preserved', () => {
        let id;
        let prior;
        let msecs = RFC_MSECS;
        for (let i = 0; i < 20000; ++i) {
            if (i % 1500 === 0) {
                msecs += 1;
            }
            id = (0, v7_js_1.default)({ msecs, seq: i });
            if (prior !== undefined) {
                assert.ok(prior < id, `${prior} < ${id}`);
            }
            prior = id;
        }
    });
    (0, node_test_1.default)('can supply seq', () => {
        let seq = 0x12345;
        let uuid = (0, v7_js_1.default)({
            msecs: RFC_MSECS,
            seq,
        });
        assert.strictEqual(uuid.substr(0, 25), '017f22e2-79b0-7000-848d-1');
        seq = 0x6fffffff;
        uuid = (0, v7_js_1.default)({
            msecs: RFC_MSECS,
            seq,
        });
        assert.strictEqual(uuid.substring(0, 25), '017f22e2-79b0-76ff-bfff-f');
    });
    (0, node_test_1.default)('internal seq is reset upon timestamp change', () => {
        (0, v7_js_1.default)({
            msecs: RFC_MSECS,
            seq: 0x6fffffff,
        });
        const uuid = (0, v7_js_1.default)({
            msecs: RFC_MSECS + 1,
        });
        assert.ok(uuid.indexOf('fff') !== 15);
    });
    (0, node_test_1.default)('v7() state transitions', () => {
        const tests = [
            {
                title: 'new time interval',
                state: { msecs: 1, seq: 123 },
                now: 2,
                expected: {
                    msecs: 2,
                    seq: 0x6c318c4,
                },
            },
            {
                title: 'same time interval',
                state: { msecs: 1, seq: 123 },
                now: 1,
                expected: {
                    msecs: 1,
                    seq: 124,
                },
            },
            {
                title: 'same time interval (sequence rollover)',
                state: { msecs: 1, seq: 0xffffffff },
                now: 1,
                expected: {
                    msecs: 2,
                    seq: 0,
                },
            },
            {
                title: 'time regression',
                state: { msecs: 2, seq: 123 },
                now: 1,
                expected: {
                    msecs: 2,
                    seq: 124,
                },
            },
            {
                title: 'time regression (sequence rollover)',
                state: { msecs: 2, seq: 0xffffffff },
                now: 1,
                expected: {
                    msecs: 3,
                    seq: 0,
                },
            },
        ];
        for (const { title, state, now, expected } of tests) {
            assert.deepStrictEqual((0, v7_js_1.updateV7State)(state, now, RFC_RANDOM), expected, `Failed: ${title}`);
        }
    });
    (0, node_test_1.default)('flipping bits changes the result', () => {
        const asBigInt = (buf) => buf.reduce((acc, v) => (acc << 8n) | BigInt(v), 0n);
        const asNumber = (bits, data) => Number(BigInt.asUintN(bits, data));
        const flip = (data, n) => data ^ (1n << BigInt(127 - n));
        const optionsFrom = (data) => {
            const ms = asNumber(48, data >> 80n);
            const hi = asNumber(12, data >> 64n);
            const lo = asNumber(20, data >> 42n);
            const r = BigInt.asUintN(42, data);
            return {
                msecs: ms,
                seq: (hi << 20) | lo,
                random: Uint8Array.from([
                    ...Array(10).fill(0),
                    ...Array(6)
                        .fill(0)
                        .map((_, i) => asNumber(8, r >> (BigInt(i) * 8n)))
                        .reverse(),
                ]),
            };
        };
        const buf = new Uint8Array(16);
        const data = asBigInt((0, v7_js_1.default)({}, buf));
        const id = (0, stringify_js_1.default)(buf);
        const reserved = [48, 49, 50, 51, 64, 65];
        for (let i = 0; i < 128; ++i) {
            if (reserved.includes(i)) {
                continue;
            }
            const flipped = flip(data, i);
            assert.strictEqual(asBigInt((0, v7_js_1.default)(optionsFrom(flipped), buf)).toString(16), flipped.toString(16), `Unequal uuids at bit ${i}`);
            assert.notStrictEqual((0, stringify_js_1.default)(buf), id);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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