"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FaqService = void 0;
const uuid_1 = require("uuid");
const database_1 = require("../application/database");
const response_error_1 = require("../error/response-error");
const faq_model_1 = require("../model/faq-model");
const faq_validation_1 = require("../validation/faq-validation");
const validation_1 = require("../validation/validation");
class FaqService {
    static create(request) {
        return __awaiter(this, void 0, void 0, function* () {
            // Validate the create request
            const createRequest = validation_1.Validation.validate(faq_validation_1.FaqValidation.CREATE, request);
            // Generate a UUID for the new faq
            const faqId = (0, uuid_1.v4)();
            // Insert the new faq into the database with the UUID
            yield database_1.db.query('INSERT INTO faq (id, question, answer) VALUES (?, ?, ?)', [
                faqId,
                createRequest.question,
                createRequest.answer,
            ]);
            // Retrieve the newly created faq by the UUID
            const faq = yield database_1.db.queryOne('SELECT * FROM faq WHERE id = ? LIMIT 1', [faqId]);
            if (!faq) {
                throw new response_error_1.ResponseError(500, 'Failed to create faq');
            }
            return (0, faq_model_1.toFaqResponse)(faq);
        });
    }
    static getAll() {
        return __awaiter(this, void 0, void 0, function* () {
            // Retrieve all faqs from the database
            const faqs = yield database_1.db.query('SELECT * FROM faq ORDER BY created_at ASC');
            // Map each faq to the response format
            return faqs.map(faq_model_1.toFaqResponse);
        });
    }
    static get(id) {
        return __awaiter(this, void 0, void 0, function* () {
            // Retrieve a single faq by ID
            const faq = yield database_1.db.queryOne('SELECT * FROM faq WHERE id = ? LIMIT 1', [id]);
            if (!faq) {
                throw new response_error_1.ResponseError(404, 'Faq not found');
            }
            return (0, faq_model_1.toFaqResponse)(faq);
        });
    }
    static update(id, request) {
        return __awaiter(this, void 0, void 0, function* () {
            // Check if the faq exists
            const existingFaq = yield database_1.db.queryOne('SELECT id FROM faq WHERE id = ? LIMIT 1', [id]);
            if (!existingFaq) {
                throw new response_error_1.ResponseError(404, 'Faq not found');
            }
            // Validate the update request
            const updateRequest = validation_1.Validation.validate(faq_validation_1.FaqValidation.UPDATE, request);
            // Update the faq in the database
            yield database_1.db.query('UPDATE faq SET question = ?, answer = ? WHERE id = ?', [updateRequest.question, updateRequest.answer, id]);
            // Retrieve the updated faq by the ID
            const faq = yield database_1.db.queryOne('SELECT * FROM faq WHERE id = ? LIMIT 1', [id]);
            if (!faq) {
                throw new response_error_1.ResponseError(500, 'Failed to update faq');
            }
            return (0, faq_model_1.toFaqResponse)(faq);
        });
    }
    static delete(id) {
        return __awaiter(this, void 0, void 0, function* () {
            // Check if the faq exists
            const existingFaq = yield database_1.db.queryOne('SELECT id FROM faq WHERE id = ? LIMIT 1', [id]);
            if (!existingFaq) {
                throw new response_error_1.ResponseError(404, 'Faq not found');
            }
            // Delete the faq from the database
            yield database_1.db.query('DELETE FROM faq WHERE id = ?', [id]);
        });
    }
}
exports.FaqService = FaqService;
