"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CategoryService = void 0;
const uuid_1 = require("uuid");
const database_1 = require("../application/database");
const response_error_1 = require("../error/response-error");
const category_model_1 = require("../model/category-model");
const category_validation_1 = require("../validation/category-validation");
const validation_1 = require("../validation/validation");
class CategoryService {
    static create(request) {
        return __awaiter(this, void 0, void 0, function* () {
            // Validate the create request
            const createRequest = validation_1.Validation.validate(category_validation_1.CategoryValidation.CREATE, request);
            // Generate a UUID for the new category
            const categoryId = (0, uuid_1.v4)();
            // Insert the new category into the database with the UUID
            yield database_1.db.query('INSERT INTO category (id, name) VALUES (?, ?)', [categoryId, createRequest.name]);
            // Retrieve the newly created category by the UUID
            const category = yield database_1.db.queryOne('SELECT * FROM category WHERE id = ? LIMIT 1', [categoryId]);
            if (!category) {
                throw new response_error_1.ResponseError(500, 'Failed to create category');
            }
            return (0, category_model_1.toCategoryResponse)(category);
        });
    }
    static getAll() {
        return __awaiter(this, void 0, void 0, function* () {
            // Retrieve all categories from the database
            const categories = yield database_1.db.query('SELECT * FROM category ORDER BY created_at ASC');
            // Map each category to the response format
            return categories.map(category_model_1.toCategoryResponse);
        });
    }
    static get(id) {
        return __awaiter(this, void 0, void 0, function* () {
            // Retrieve a single category by ID
            const category = yield database_1.db.queryOne('SELECT * FROM category WHERE id = ? LIMIT 1', [id]);
            if (!category) {
                throw new response_error_1.ResponseError(404, 'Category not found');
            }
            return (0, category_model_1.toCategoryResponse)(category);
        });
    }
    static update(id, request) {
        return __awaiter(this, void 0, void 0, function* () {
            // Check if the category exists
            const existingCategory = yield database_1.db.queryOne('SELECT id FROM category WHERE id = ? LIMIT 1', [id]);
            if (!existingCategory) {
                throw new response_error_1.ResponseError(404, 'Category not found');
            }
            // Validate the update request
            const updateRequest = validation_1.Validation.validate(category_validation_1.CategoryValidation.CREATE, request);
            // Update the category in the database
            yield database_1.db.query('UPDATE category SET name = ? WHERE id = ?', [updateRequest.name, id]);
            // Retrieve the updated category by ID
            const category = yield database_1.db.queryOne('SELECT * FROM category WHERE id = ?', [id]);
            if (!category) {
                throw new response_error_1.ResponseError(500, 'Failed to update category');
            }
            return (0, category_model_1.toCategoryResponse)(category);
        });
    }
    static delete(id) {
        return __awaiter(this, void 0, void 0, function* () {
            // Retrieve the category before deleting it
            const category = yield database_1.db.queryOne('SELECT * FROM category WHERE id = ?', [id]);
            if (!category) {
                throw new response_error_1.ResponseError(404, 'Category not found');
            }
            // Delete the category from the database
            yield database_1.db.query('DELETE FROM category WHERE id = ?', [id]);
            return (0, category_model_1.toCategoryResponse)(category);
        });
    }
}
exports.CategoryService = CategoryService;
